'use strict';

/* --------------------------------------------------------------
 datatable_fixed_header.js 2016-07-13
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Enable Fixed DataTable Header
 *
 * The table header will remain in the viewport as the user scrolls down the page. The style change of this
 * module is a bit tricky because we need to remove the thead from the normal flow, something that breaks the
 * display of the table. Therefore a helper clone of the thead is used to maintain the table formatting.
 *
 * **Notice #1**: The .table-fixed-header class is styled by the _tables.scss and is part of this solution.
 *
 * **Notice #2**: This method will take into concern the .content-header element which shouldn't overlap the
 * table header.
 *
 * @module Admin/Extensions/datatable_fixed_header
 */
gx.extensions.module('datatable_fixed_header', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Table Header Selector
     *
     * @type {jQuery}
     */
    var $thead = $this.children('thead');

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    /**
     * Marks the end of the table.
     *
     * This value is used to stop the fixed header when the user reaches the end of the table.
     *
     * @type {Number}
     */
    var tableOffsetBottom = void 0;

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * On DataTable Draw Event
     *
     * Re-calculate the table bottom offset value.
     */
    function _onDataTableDraw() {
        tableOffsetBottom = $this.offset().top + $this.height() - $thead.height();
    }

    /**
     * On DataTable Initialization
     *
     * Modify the table HTML and set the required event handling for the fixed header functionality.
     */
    function _onDataTableInit() {
        var $mainHeader = $('#main-header');
        var $contentHeader = $('.content-header');
        var $clone = $thead.clone();
        var originalTop = $thead.offset().top;
        var isFixed = false;
        var rollingAnimationInterval = null;

        $clone.hide().addClass('table-fixed-header-helper').prependTo($this);

        $(window).on('scroll', function () {
            var scrollTop = $(window).scrollTop();

            if (!isFixed && scrollTop + $mainHeader.outerHeight() > originalTop) {
                $this.addClass('table-fixed-header');
                $thead.outerWidth($this.outerWidth()).addClass('fixed');
                $clone.show();
                isFixed = true;
            } else if (isFixed && scrollTop + $mainHeader.outerHeight() < originalTop) {
                $this.removeClass('table-fixed-header');
                $thead.outerWidth('').removeClass('fixed');
                $clone.hide();
                isFixed = false;
            }

            if (scrollTop >= tableOffsetBottom) {
                $thead.removeClass('fixed');
            } else if ($(window).scrollTop() < tableOffsetBottom && !$thead.hasClass('fixed')) {
                $thead.addClass('fixed');
            }
        }).on('content_header:roll_in', function () {
            rollingAnimationInterval = setInterval(function () {
                $thead.css('top', $contentHeader.position().top + $contentHeader.outerHeight());
                if ($contentHeader.hasClass('fixed')) {
                    clearInterval(rollingAnimationInterval);
                }
            }, 1);
        }).on('content_header:roll_out', function () {
            clearInterval(rollingAnimationInterval);
            $thead.css('top', $mainHeader.outerHeight());
        });
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $(window).on('JSENGINE_INIT_FINISHED', function () {
            $this.on('draw.dt', _onDataTableDraw).on('init.dt', _onDataTableInit);

            // Setup fixed header functionality if the table is already initialized.
            if ($this.DataTable().ajax.json() !== undefined) {
                _onDataTableInit();
            }
        });

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImRhdGF0YWJsZV9maXhlZF9oZWFkZXIuanMiXSwibmFtZXMiOlsiZ3giLCJleHRlbnNpb25zIiwibW9kdWxlIiwiZGF0YSIsIiR0aGlzIiwiJCIsIiR0aGVhZCIsImNoaWxkcmVuIiwidGFibGVPZmZzZXRCb3R0b20iLCJfb25EYXRhVGFibGVEcmF3Iiwib2Zmc2V0IiwidG9wIiwiaGVpZ2h0IiwiX29uRGF0YVRhYmxlSW5pdCIsIiRtYWluSGVhZGVyIiwiJGNvbnRlbnRIZWFkZXIiLCIkY2xvbmUiLCJjbG9uZSIsIm9yaWdpbmFsVG9wIiwiaXNGaXhlZCIsInJvbGxpbmdBbmltYXRpb25JbnRlcnZhbCIsImhpZGUiLCJhZGRDbGFzcyIsInByZXBlbmRUbyIsIndpbmRvdyIsIm9uIiwic2Nyb2xsVG9wIiwib3V0ZXJIZWlnaHQiLCJvdXRlcldpZHRoIiwic2hvdyIsInJlbW92ZUNsYXNzIiwiaGFzQ2xhc3MiLCJzZXRJbnRlcnZhbCIsImNzcyIsInBvc2l0aW9uIiwiY2xlYXJJbnRlcnZhbCIsImluaXQiLCJkb25lIiwiRGF0YVRhYmxlIiwiYWpheCIsImpzb24iLCJ1bmRlZmluZWQiXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7Ozs7QUFVQTs7Ozs7Ozs7Ozs7Ozs7QUFjQUEsR0FBR0MsVUFBSCxDQUFjQyxNQUFkLENBQXFCLHdCQUFyQixFQUErQyxFQUEvQyxFQUFtRCxVQUFVQyxJQUFWLEVBQWdCOztBQUUvRDs7QUFFQTtBQUNBO0FBQ0E7O0FBRUE7Ozs7OztBQUtBLFFBQU1DLFFBQVFDLEVBQUUsSUFBRixDQUFkOztBQUVBOzs7OztBQUtBLFFBQU1DLFNBQVNGLE1BQU1HLFFBQU4sQ0FBZSxPQUFmLENBQWY7O0FBRUE7Ozs7O0FBS0EsUUFBTUwsU0FBUyxFQUFmOztBQUVBOzs7Ozs7O0FBT0EsUUFBSU0sMEJBQUo7O0FBRUE7QUFDQTtBQUNBOztBQUVBOzs7OztBQUtBLGFBQVNDLGdCQUFULEdBQTRCO0FBQ3hCRCw0QkFBb0JKLE1BQU1NLE1BQU4sR0FBZUMsR0FBZixHQUFxQlAsTUFBTVEsTUFBTixFQUFyQixHQUFzQ04sT0FBT00sTUFBUCxFQUExRDtBQUNIOztBQUVEOzs7OztBQUtBLGFBQVNDLGdCQUFULEdBQTRCO0FBQ3hCLFlBQU1DLGNBQWNULEVBQUUsY0FBRixDQUFwQjtBQUNBLFlBQU1VLGlCQUFpQlYsRUFBRSxpQkFBRixDQUF2QjtBQUNBLFlBQU1XLFNBQVNWLE9BQU9XLEtBQVAsRUFBZjtBQUNBLFlBQU1DLGNBQWNaLE9BQU9JLE1BQVAsR0FBZ0JDLEdBQXBDO0FBQ0EsWUFBSVEsVUFBVSxLQUFkO0FBQ0EsWUFBSUMsMkJBQTJCLElBQS9COztBQUVBSixlQUNLSyxJQURMLEdBRUtDLFFBRkwsQ0FFYywyQkFGZCxFQUdLQyxTQUhMLENBR2VuQixLQUhmOztBQUtBQyxVQUFFbUIsTUFBRixFQUNLQyxFQURMLENBQ1EsUUFEUixFQUNrQixZQUFZO0FBQ3RCLGdCQUFNQyxZQUFZckIsRUFBRW1CLE1BQUYsRUFBVUUsU0FBVixFQUFsQjs7QUFFQSxnQkFBSSxDQUFDUCxPQUFELElBQVlPLFlBQVlaLFlBQVlhLFdBQVosRUFBWixHQUF3Q1QsV0FBeEQsRUFBcUU7QUFDakVkLHNCQUFNa0IsUUFBTixDQUFlLG9CQUFmO0FBQ0FoQix1QkFDS3NCLFVBREwsQ0FDZ0J4QixNQUFNd0IsVUFBTixFQURoQixFQUVLTixRQUZMLENBRWMsT0FGZDtBQUdBTix1QkFBT2EsSUFBUDtBQUNBViwwQkFBVSxJQUFWO0FBQ0gsYUFQRCxNQU9PLElBQUlBLFdBQVdPLFlBQVlaLFlBQVlhLFdBQVosRUFBWixHQUF3Q1QsV0FBdkQsRUFBb0U7QUFDdkVkLHNCQUFNMEIsV0FBTixDQUFrQixvQkFBbEI7QUFDQXhCLHVCQUNLc0IsVUFETCxDQUNnQixFQURoQixFQUVLRSxXQUZMLENBRWlCLE9BRmpCO0FBR0FkLHVCQUFPSyxJQUFQO0FBQ0FGLDBCQUFVLEtBQVY7QUFDSDs7QUFFRCxnQkFBSU8sYUFBYWxCLGlCQUFqQixFQUFvQztBQUNoQ0YsdUJBQU93QixXQUFQLENBQW1CLE9BQW5CO0FBQ0gsYUFGRCxNQUVPLElBQUl6QixFQUFFbUIsTUFBRixFQUFVRSxTQUFWLEtBQXdCbEIsaUJBQXhCLElBQTZDLENBQUNGLE9BQU95QixRQUFQLENBQWdCLE9BQWhCLENBQWxELEVBQTRFO0FBQy9FekIsdUJBQU9nQixRQUFQLENBQWdCLE9BQWhCO0FBQ0g7QUFDSixTQXpCTCxFQTBCS0csRUExQkwsQ0EwQlEsd0JBMUJSLEVBMEJrQyxZQUFZO0FBQ3RDTCx1Q0FBMkJZLFlBQVksWUFBTTtBQUN6QzFCLHVCQUFPMkIsR0FBUCxDQUFXLEtBQVgsRUFBa0JsQixlQUFlbUIsUUFBZixHQUEwQnZCLEdBQTFCLEdBQWdDSSxlQUFlWSxXQUFmLEVBQWxEO0FBQ0Esb0JBQUlaLGVBQWVnQixRQUFmLENBQXdCLE9BQXhCLENBQUosRUFBc0M7QUFDbENJLGtDQUFjZix3QkFBZDtBQUNIO0FBQ0osYUFMMEIsRUFLeEIsQ0FMd0IsQ0FBM0I7QUFNSCxTQWpDTCxFQWtDS0ssRUFsQ0wsQ0FrQ1EseUJBbENSLEVBa0NtQyxZQUFZO0FBQ3ZDVSwwQkFBY2Ysd0JBQWQ7QUFDQWQsbUJBQU8yQixHQUFQLENBQVcsS0FBWCxFQUFrQm5CLFlBQVlhLFdBQVosRUFBbEI7QUFDSCxTQXJDTDtBQXNDSDs7QUFHRDtBQUNBO0FBQ0E7O0FBRUF6QixXQUFPa0MsSUFBUCxHQUFjLFVBQVVDLElBQVYsRUFBZ0I7QUFDMUJoQyxVQUFFbUIsTUFBRixFQUFVQyxFQUFWLENBQWEsd0JBQWIsRUFBdUMsWUFBTTtBQUN6Q3JCLGtCQUNLcUIsRUFETCxDQUNRLFNBRFIsRUFDbUJoQixnQkFEbkIsRUFFS2dCLEVBRkwsQ0FFUSxTQUZSLEVBRW1CWixnQkFGbkI7O0FBSUE7QUFDQSxnQkFBSVQsTUFBTWtDLFNBQU4sR0FBa0JDLElBQWxCLENBQXVCQyxJQUF2QixPQUFrQ0MsU0FBdEMsRUFBaUQ7QUFDN0M1QjtBQUNIO0FBQ0osU0FURDs7QUFXQXdCO0FBQ0gsS0FiRDs7QUFlQSxXQUFPbkMsTUFBUDtBQUVILENBbklEIiwiZmlsZSI6ImRhdGF0YWJsZV9maXhlZF9oZWFkZXIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxyXG4gZGF0YXRhYmxlX2ZpeGVkX2hlYWRlci5qcyAyMDE2LTA3LTEzXHJcbiBHYW1iaW8gR21iSFxyXG4gaHR0cDovL3d3dy5nYW1iaW8uZGVcclxuIENvcHlyaWdodCAoYykgMjAxNiBHYW1iaW8gR21iSFxyXG4gUmVsZWFzZWQgdW5kZXIgdGhlIEdOVSBHZW5lcmFsIFB1YmxpYyBMaWNlbnNlIChWZXJzaW9uIDIpXHJcbiBbaHR0cDovL3d3dy5nbnUub3JnL2xpY2Vuc2VzL2dwbC0yLjAuaHRtbF1cclxuIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiAqL1xyXG5cclxuLyoqXHJcbiAqICMjIEVuYWJsZSBGaXhlZCBEYXRhVGFibGUgSGVhZGVyXHJcbiAqXHJcbiAqIFRoZSB0YWJsZSBoZWFkZXIgd2lsbCByZW1haW4gaW4gdGhlIHZpZXdwb3J0IGFzIHRoZSB1c2VyIHNjcm9sbHMgZG93biB0aGUgcGFnZS4gVGhlIHN0eWxlIGNoYW5nZSBvZiB0aGlzXHJcbiAqIG1vZHVsZSBpcyBhIGJpdCB0cmlja3kgYmVjYXVzZSB3ZSBuZWVkIHRvIHJlbW92ZSB0aGUgdGhlYWQgZnJvbSB0aGUgbm9ybWFsIGZsb3csIHNvbWV0aGluZyB0aGF0IGJyZWFrcyB0aGVcclxuICogZGlzcGxheSBvZiB0aGUgdGFibGUuIFRoZXJlZm9yZSBhIGhlbHBlciBjbG9uZSBvZiB0aGUgdGhlYWQgaXMgdXNlZCB0byBtYWludGFpbiB0aGUgdGFibGUgZm9ybWF0dGluZy5cclxuICpcclxuICogKipOb3RpY2UgIzEqKjogVGhlIC50YWJsZS1maXhlZC1oZWFkZXIgY2xhc3MgaXMgc3R5bGVkIGJ5IHRoZSBfdGFibGVzLnNjc3MgYW5kIGlzIHBhcnQgb2YgdGhpcyBzb2x1dGlvbi5cclxuICpcclxuICogKipOb3RpY2UgIzIqKjogVGhpcyBtZXRob2Qgd2lsbCB0YWtlIGludG8gY29uY2VybiB0aGUgLmNvbnRlbnQtaGVhZGVyIGVsZW1lbnQgd2hpY2ggc2hvdWxkbid0IG92ZXJsYXAgdGhlXHJcbiAqIHRhYmxlIGhlYWRlci5cclxuICpcclxuICogQG1vZHVsZSBBZG1pbi9FeHRlbnNpb25zL2RhdGF0YWJsZV9maXhlZF9oZWFkZXJcclxuICovXHJcbmd4LmV4dGVuc2lvbnMubW9kdWxlKCdkYXRhdGFibGVfZml4ZWRfaGVhZGVyJywgW10sIGZ1bmN0aW9uIChkYXRhKSB7XHJcblxyXG4gICAgJ3VzZSBzdHJpY3QnO1xyXG5cclxuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxyXG4gICAgLy8gVkFSSUFCTEVTXHJcbiAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cclxuXHJcbiAgICAvKipcclxuICAgICAqIE1vZHVsZSBTZWxlY3RvclxyXG4gICAgICpcclxuICAgICAqIEB0eXBlIHtqUXVlcnl9XHJcbiAgICAgKi9cclxuICAgIGNvbnN0ICR0aGlzID0gJCh0aGlzKTtcclxuXHJcbiAgICAvKipcclxuICAgICAqIFRhYmxlIEhlYWRlciBTZWxlY3RvclxyXG4gICAgICpcclxuICAgICAqIEB0eXBlIHtqUXVlcnl9XHJcbiAgICAgKi9cclxuICAgIGNvbnN0ICR0aGVhZCA9ICR0aGlzLmNoaWxkcmVuKCd0aGVhZCcpO1xyXG5cclxuICAgIC8qKlxyXG4gICAgICogTW9kdWxlIEluc3RhbmNlXHJcbiAgICAgKlxyXG4gICAgICogQHR5cGUge09iamVjdH1cclxuICAgICAqL1xyXG4gICAgY29uc3QgbW9kdWxlID0ge307XHJcblxyXG4gICAgLyoqXHJcbiAgICAgKiBNYXJrcyB0aGUgZW5kIG9mIHRoZSB0YWJsZS5cclxuICAgICAqXHJcbiAgICAgKiBUaGlzIHZhbHVlIGlzIHVzZWQgdG8gc3RvcCB0aGUgZml4ZWQgaGVhZGVyIHdoZW4gdGhlIHVzZXIgcmVhY2hlcyB0aGUgZW5kIG9mIHRoZSB0YWJsZS5cclxuICAgICAqXHJcbiAgICAgKiBAdHlwZSB7TnVtYmVyfVxyXG4gICAgICovXHJcbiAgICBsZXQgdGFibGVPZmZzZXRCb3R0b207XHJcblxyXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiAgICAvLyBGVU5DVElPTlNcclxuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxyXG5cclxuICAgIC8qKlxyXG4gICAgICogT24gRGF0YVRhYmxlIERyYXcgRXZlbnRcclxuICAgICAqXHJcbiAgICAgKiBSZS1jYWxjdWxhdGUgdGhlIHRhYmxlIGJvdHRvbSBvZmZzZXQgdmFsdWUuXHJcbiAgICAgKi9cclxuICAgIGZ1bmN0aW9uIF9vbkRhdGFUYWJsZURyYXcoKSB7XHJcbiAgICAgICAgdGFibGVPZmZzZXRCb3R0b20gPSAkdGhpcy5vZmZzZXQoKS50b3AgKyAkdGhpcy5oZWlnaHQoKSAtICR0aGVhZC5oZWlnaHQoKTtcclxuICAgIH1cclxuXHJcbiAgICAvKipcclxuICAgICAqIE9uIERhdGFUYWJsZSBJbml0aWFsaXphdGlvblxyXG4gICAgICpcclxuICAgICAqIE1vZGlmeSB0aGUgdGFibGUgSFRNTCBhbmQgc2V0IHRoZSByZXF1aXJlZCBldmVudCBoYW5kbGluZyBmb3IgdGhlIGZpeGVkIGhlYWRlciBmdW5jdGlvbmFsaXR5LlxyXG4gICAgICovXHJcbiAgICBmdW5jdGlvbiBfb25EYXRhVGFibGVJbml0KCkge1xyXG4gICAgICAgIGNvbnN0ICRtYWluSGVhZGVyID0gJCgnI21haW4taGVhZGVyJyk7XHJcbiAgICAgICAgY29uc3QgJGNvbnRlbnRIZWFkZXIgPSAkKCcuY29udGVudC1oZWFkZXInKTtcclxuICAgICAgICBjb25zdCAkY2xvbmUgPSAkdGhlYWQuY2xvbmUoKTtcclxuICAgICAgICBjb25zdCBvcmlnaW5hbFRvcCA9ICR0aGVhZC5vZmZzZXQoKS50b3A7XHJcbiAgICAgICAgbGV0IGlzRml4ZWQgPSBmYWxzZTtcclxuICAgICAgICBsZXQgcm9sbGluZ0FuaW1hdGlvbkludGVydmFsID0gbnVsbDtcclxuXHJcbiAgICAgICAgJGNsb25lXHJcbiAgICAgICAgICAgIC5oaWRlKClcclxuICAgICAgICAgICAgLmFkZENsYXNzKCd0YWJsZS1maXhlZC1oZWFkZXItaGVscGVyJylcclxuICAgICAgICAgICAgLnByZXBlbmRUbygkdGhpcyk7XHJcblxyXG4gICAgICAgICQod2luZG93KVxyXG4gICAgICAgICAgICAub24oJ3Njcm9sbCcsIGZ1bmN0aW9uICgpIHtcclxuICAgICAgICAgICAgICAgIGNvbnN0IHNjcm9sbFRvcCA9ICQod2luZG93KS5zY3JvbGxUb3AoKTtcclxuXHJcbiAgICAgICAgICAgICAgICBpZiAoIWlzRml4ZWQgJiYgc2Nyb2xsVG9wICsgJG1haW5IZWFkZXIub3V0ZXJIZWlnaHQoKSA+IG9yaWdpbmFsVG9wKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgJHRoaXMuYWRkQ2xhc3MoJ3RhYmxlLWZpeGVkLWhlYWRlcicpO1xyXG4gICAgICAgICAgICAgICAgICAgICR0aGVhZFxyXG4gICAgICAgICAgICAgICAgICAgICAgICAub3V0ZXJXaWR0aCgkdGhpcy5vdXRlcldpZHRoKCkpXHJcbiAgICAgICAgICAgICAgICAgICAgICAgIC5hZGRDbGFzcygnZml4ZWQnKTtcclxuICAgICAgICAgICAgICAgICAgICAkY2xvbmUuc2hvdygpO1xyXG4gICAgICAgICAgICAgICAgICAgIGlzRml4ZWQgPSB0cnVlO1xyXG4gICAgICAgICAgICAgICAgfSBlbHNlIGlmIChpc0ZpeGVkICYmIHNjcm9sbFRvcCArICRtYWluSGVhZGVyLm91dGVySGVpZ2h0KCkgPCBvcmlnaW5hbFRvcCkge1xyXG4gICAgICAgICAgICAgICAgICAgICR0aGlzLnJlbW92ZUNsYXNzKCd0YWJsZS1maXhlZC1oZWFkZXInKTtcclxuICAgICAgICAgICAgICAgICAgICAkdGhlYWRcclxuICAgICAgICAgICAgICAgICAgICAgICAgLm91dGVyV2lkdGgoJycpXHJcbiAgICAgICAgICAgICAgICAgICAgICAgIC5yZW1vdmVDbGFzcygnZml4ZWQnKTtcclxuICAgICAgICAgICAgICAgICAgICAkY2xvbmUuaGlkZSgpO1xyXG4gICAgICAgICAgICAgICAgICAgIGlzRml4ZWQgPSBmYWxzZTtcclxuICAgICAgICAgICAgICAgIH1cclxuXHJcbiAgICAgICAgICAgICAgICBpZiAoc2Nyb2xsVG9wID49IHRhYmxlT2Zmc2V0Qm90dG9tKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgJHRoZWFkLnJlbW92ZUNsYXNzKCdmaXhlZCcpO1xyXG4gICAgICAgICAgICAgICAgfSBlbHNlIGlmICgkKHdpbmRvdykuc2Nyb2xsVG9wKCkgPCB0YWJsZU9mZnNldEJvdHRvbSAmJiAhJHRoZWFkLmhhc0NsYXNzKCdmaXhlZCcpKSB7XHJcbiAgICAgICAgICAgICAgICAgICAgJHRoZWFkLmFkZENsYXNzKCdmaXhlZCcpO1xyXG4gICAgICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICB9KVxyXG4gICAgICAgICAgICAub24oJ2NvbnRlbnRfaGVhZGVyOnJvbGxfaW4nLCBmdW5jdGlvbiAoKSB7XHJcbiAgICAgICAgICAgICAgICByb2xsaW5nQW5pbWF0aW9uSW50ZXJ2YWwgPSBzZXRJbnRlcnZhbCgoKSA9PiB7XHJcbiAgICAgICAgICAgICAgICAgICAgJHRoZWFkLmNzcygndG9wJywgJGNvbnRlbnRIZWFkZXIucG9zaXRpb24oKS50b3AgKyAkY29udGVudEhlYWRlci5vdXRlckhlaWdodCgpKTtcclxuICAgICAgICAgICAgICAgICAgICBpZiAoJGNvbnRlbnRIZWFkZXIuaGFzQ2xhc3MoJ2ZpeGVkJykpIHtcclxuICAgICAgICAgICAgICAgICAgICAgICAgY2xlYXJJbnRlcnZhbChyb2xsaW5nQW5pbWF0aW9uSW50ZXJ2YWwpO1xyXG4gICAgICAgICAgICAgICAgICAgIH1cclxuICAgICAgICAgICAgICAgIH0sIDEpO1xyXG4gICAgICAgICAgICB9KVxyXG4gICAgICAgICAgICAub24oJ2NvbnRlbnRfaGVhZGVyOnJvbGxfb3V0JywgZnVuY3Rpb24gKCkge1xyXG4gICAgICAgICAgICAgICAgY2xlYXJJbnRlcnZhbChyb2xsaW5nQW5pbWF0aW9uSW50ZXJ2YWwpO1xyXG4gICAgICAgICAgICAgICAgJHRoZWFkLmNzcygndG9wJywgJG1haW5IZWFkZXIub3V0ZXJIZWlnaHQoKSk7XHJcbiAgICAgICAgICAgIH0pO1xyXG4gICAgfVxyXG5cclxuXHJcbiAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cclxuICAgIC8vIElOSVRJQUxJWkFUSU9OXHJcbiAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cclxuXHJcbiAgICBtb2R1bGUuaW5pdCA9IGZ1bmN0aW9uIChkb25lKSB7XHJcbiAgICAgICAgJCh3aW5kb3cpLm9uKCdKU0VOR0lORV9JTklUX0ZJTklTSEVEJywgKCkgPT4ge1xyXG4gICAgICAgICAgICAkdGhpc1xyXG4gICAgICAgICAgICAgICAgLm9uKCdkcmF3LmR0JywgX29uRGF0YVRhYmxlRHJhdylcclxuICAgICAgICAgICAgICAgIC5vbignaW5pdC5kdCcsIF9vbkRhdGFUYWJsZUluaXQpO1xyXG5cclxuICAgICAgICAgICAgLy8gU2V0dXAgZml4ZWQgaGVhZGVyIGZ1bmN0aW9uYWxpdHkgaWYgdGhlIHRhYmxlIGlzIGFscmVhZHkgaW5pdGlhbGl6ZWQuXHJcbiAgICAgICAgICAgIGlmICgkdGhpcy5EYXRhVGFibGUoKS5hamF4Lmpzb24oKSAhPT0gdW5kZWZpbmVkKSB7XHJcbiAgICAgICAgICAgICAgICBfb25EYXRhVGFibGVJbml0KCk7XHJcbiAgICAgICAgICAgIH1cclxuICAgICAgICB9KTtcclxuXHJcbiAgICAgICAgZG9uZSgpO1xyXG4gICAgfTtcclxuXHJcbiAgICByZXR1cm4gbW9kdWxlO1xyXG5cclxufSk7Il19
